<?php
if (!defined('ABSPATH')) exit;

class WWNT_Notifications {
    private static $instance = null;
    private $api_handler;

    private function __construct() {
        $this->api_handler = WWNT_API_Handler::get_instance();
        
        add_action('woocommerce_order_status_changed', [$this, 'handle_order_status_change'], 10, 4);
        add_action('woocommerce_after_order_object_save', [$this, 'maybe_handle_status_change'], 10, 2);
        add_action('wp_ajax_wwnt_send_test_notification', [$this, 'send_test_notification']);
    }
    
    public function maybe_handle_status_change($order, $data_store) {
        if (!empty($order->get_changes()['status'])) {
            $old_status = $order->get_status($context = 'edit');
            $new_status = $order->get_status();
            $this->handle_order_status_change($order->get_id(), $old_status, $new_status, $order);
        }
    }

    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

   public function handle_order_status_change($order_id, $old_status, $new_status, $order) {
   
    $settings = get_option('wwnt_settings');
    
    if (empty($settings['enable_' . $new_status])) {
        return;
    }

        $recipient = $settings['recipient_' . $new_status] ?? 'customer';
        $custom_recipient = $settings['custom_recipient_' . $new_status] ?? '';
        
        $recipients = [];
        
        if ($recipient === 'customer' || $recipient === 'both') {
            $phone = $order->get_billing_phone();
            if ($phone) {
                $recipients[] = $phone;
            }
        }
        
        if ($recipient === 'admin' || $recipient === 'both') {
            $admin_phone = $settings['admin_phone'] ?? '';
            if ($admin_phone) {
                $recipients[] = $admin_phone;
            }
        }
        
        if ($recipient === 'custom' && $custom_recipient) {
            $recipients[] = $custom_recipient;
        }
        
        if (empty($recipients)) {
            return;
        }
        
        $template = $settings['template_' . $new_status] ?? '';
        if (empty($template)) {
            return;
        }
        
        $message = $this->replace_placeholders($template, $order);
        $from_number = $settings['from_number'] ?? '';
        
        foreach ($recipients as $to) {
            $this->send_message($to, $from_number, $message);
        }
    }

    private function replace_placeholders($template, $order) {
    $product_names = [];
    $product_names_br = [];
    $product_links = [];
    $product_links_br = [];
    $product_name_links = [];
    $product_name_links_br = [];
    $product_descriptions = [];
    $product_excerpts = [];
    $product_skus = [];
    $product_skus_br = [];
    $order_items = [];
    $order_items_br = [];
    $order_items_price = [];
    $order_items_price_br = [];
    $all_order_items_br = [];
    $variable_names = [];
    $variable_names_br = [];
    $variable_items = [];
    $variable_items_br = [];

    $total_products = 0;
    $total_items = 0;

    foreach ($order->get_items() as $item) {
        $product = $item->get_product();
        if (!$product) continue;

        $total_products++;
        $total_items += $item->get_quantity();

        $product_names[]     = $product->get_name();
        $product_names_br[]  = $product->get_name();
        $product_descriptions[] = $product->get_description();
        $product_excerpts[]     = $product->get_short_description();

        $sku = $product->get_sku();
        if ($sku) {
            $product_skus[]    = $sku;
            $product_skus_br[] = $sku;
        }

        $product_link = get_permalink($product->get_id());
        $product_links[]    = $product_link;
        $product_links_br[] = $product_link;

        $product_name_links[]    = $product->get_name() . ' - ' . $product_link;
        $product_name_links_br[] = $product->get_name() . ' - ' . $product_link;

        $order_items[]       = $item->get_name() . ' × ' . $item->get_quantity();
        $order_items_br[]    = $item->get_name() . ' × ' . $item->get_quantity();
        $order_items_price[] = $item->get_name() . ' × ' . $item->get_quantity() . ' = ' . wc_price($item->get_total());
        $order_items_price_br[] = $item->get_name() . ' × ' . $item->get_quantity() . ' = ' . wc_price($item->get_total());
        $all_order_items_br[]   = $item->get_name() . ' × ' . $item->get_quantity() . ' = ' . wc_price($item->get_total());

        if ($product->is_type('variation')) {
            $variable_names[]     = $product->get_name();
            $variable_names_br[]  = $product->get_name();
            $variable_items[]     = $product->get_name() . ' × ' . $item->get_quantity();
            $variable_items_br[]  = $product->get_name() . ' × ' . $item->get_quantity();
        }
    }

    $replacements = [

        // عام (General)
        '{{sitename}}'       => get_bloginfo('name'),
        '{{siteurl}}'        => get_site_url(),
        '{{wordpress-url}}'  => admin_url(),
        '{{tagline}}'        => get_bloginfo('description'),
        '{{privacy-policy}}' => function_exists('get_privacy_policy_url') ? get_privacy_policy_url() : '',

        // تفاصيل حساب ووردبريس
        '{{wp-first-name}}'   => wp_get_current_user()->first_name ?? '',
        '{{wp-last-name}}'    => wp_get_current_user()->last_name ?? '',
        '{{wp-username}}'     => wp_get_current_user()->user_login ?? '',
        '{{wp-nickname}}'     => wp_get_current_user()->nickname ?? '',
        '{{wp-display-name}}' => wp_get_current_user()->display_name ?? '',
        '{{wp-email}}'        => wp_get_current_user()->user_email ?? '',
        '{{wp-user-website}}' => wp_get_current_user()->user_url ?? '',
        '{{wp-user-bio}}'     => get_user_meta(get_current_user_id(), 'description', true),

        // تفاصيل الطلب في ووكومرس
        '{{wc-order}}'                   => $order->get_order_number(),
        '{{wc-order-id}}'                => $order->get_id(),
        '{{wc-order-key}}'               => $order->get_order_key(),
        '{{wc-order-status}}'            => wc_get_order_status_name($order->get_status()),
        '{{wc-order-date}}'              => wc_format_datetime($order->get_date_created()),
        '{{wc-product-names}}'           => implode(', ', $product_names),
        '{{wc-product-names-br}}'        => implode("\n", $product_names_br),
        '{{wc-product-names-variable}}'  => implode(', ', $variable_names),
        '{{wc-product-names-variable-br}}' => implode("\n", $variable_names_br),
        '{{wc-product-description}}'     => implode("\n", $product_descriptions),
        '{{wc-product-excerpt}}'         => implode("\n", $product_excerpts),
        '{{wc-product-name-count}}'      => !empty($product_names) ? $product_names[0] . ' +' . ($total_products - 1) : '',
        '{{wc-product-link}}'            => implode(', ', $product_links),
        '{{wc-product-link-br}}'         => implode("\n", $product_links_br),
        '{{wc-product-name-link}}'       => implode(', ', $product_name_links),
        '{{wc-product-name-link-br}}'    => implode("\n", $product_name_links_br),
        '{{wc-total-products}}'          => $total_products,
        '{{wc-total-items}}'             => $total_items,
        '{{wc-order-items}}'             => implode(', ', $order_items),
        '{{wc-order-items-br}}'          => implode("\n", $order_items_br),
        '{{wc-order-items-variable}}'    => implode(', ', $variable_items),
        '{{wc-order-items-variable-br}}' => implode("\n", $variable_items_br),
        '{{wc-order-items-price}}'       => implode(', ', $order_items_price),
        '{{wc-order-items-price-br}}'    => implode("\n", $order_items_price_br),
        '{{wc-all-order-items-br}}'      => implode("\n", $all_order_items_br),
        '{{wc-sku}}'                     => implode(', ', $product_skus),
        '{{wc-sku-br}}'                  => implode("\n", $product_skus_br),
        '{{wc-order-amount}}'            => $order->get_formatted_order_total(),
        '{{wc-discount}}'                => wc_price($order->get_discount_total()),
        '{{wc-tax}}'                     => wc_price($order->get_total_tax()),
        '{{wc-order-amount-ex-tax}}'     => wc_price($order->get_total() - $order->get_total_tax()),
        '{{wc-payment-method}}'          => $order->get_payment_method_title(),
        '{{wc-transaction-id}}'          => $order->get_transaction_id(),
        '{{wc-shipping-method}}'         => $order->get_shipping_method(),
        '{{wc-shipping-cost}}'           => wc_price($order->get_shipping_total()),
        '{{wc-refund-amount}}'           => wc_price($order->get_total_refunded()),
        '{{wc-refund-reason}}'           => '', // يحتاج إضافة يدوي إذا فيه تخزين للسبب
        '{{wc-order-notes}}'             => $order->get_customer_note(),

        // روابط الطلب
        '{{wc-order-confirmation-link}}' => $order->get_checkout_order_received_url(),
        '{{wc-order-pay-link}}'          => $order->get_checkout_payment_url(),

        // تفاصيل الفوترة
        '{{wc-billing-first-name}}'     => $order->get_billing_first_name(),
        '{{wc-billing-last-name}}'      => $order->get_billing_last_name(),
        '{{wc-billing-company}}'        => $order->get_billing_company(),
        '{{wc-billing-address-line-1}}' => $order->get_billing_address_1(),
        '{{wc-billing-address-line-2}}' => $order->get_billing_address_2(),
        '{{wc-billing-city}}'           => $order->get_billing_city(),
        '{{wc-billing-postcode}}'       => $order->get_billing_postcode(),
        '{{wc-billing-state}}'          => $order->get_billing_state(),
        '{{wc-billing-country}}'        => $order->get_billing_country(),
        '{{wc-billing-email}}'          => $order->get_billing_email(),
        '{{wc-billing-phone}}'          => $order->get_billing_phone(),

        // تفاصيل الشحن
        '{{wc-shipping-first-name}}'     => $order->get_shipping_first_name(),
        '{{wc-shipping-last-name}}'      => $order->get_shipping_last_name(),
        '{{wc-shipping-company}}'        => $order->get_shipping_company(),
        '{{wc-shipping-address-line-1}}' => $order->get_shipping_address_1(),
        '{{wc-shipping-address-line-2}}' => $order->get_shipping_address_2(),
        '{{wc-shipping-city}}'           => $order->get_shipping_city(),
        '{{wc-shipping-postcode}}'       => $order->get_shipping_postcode(),
        '{{wc-shipping-state}}'          => $order->get_shipping_state(),
        '{{wc-shipping-country}}'        => $order->get_shipping_country(),

        // التنزيلات الرقمية (EDD)
        '{{edd-download_list}}' => '', // تحتاج ربط مع EDD لو مستخدمه
    ];

    return str_replace(array_keys($replacements), array_values($replacements), $template);

}


    private function send_message($to, $from, $message) {
        $message_data = [
            'from' => $from,
            'text' => $message,
        ];
        return $this->api_handler->send_whatsapp_message($to, 'text', $message_data);
    }

    public function send_test_notification() {
        check_ajax_referer('wwnt_test_nonce', 'security');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('لا تملك صلاحية لإرسال إشعارات اختبار', 'woocommerce-whatsapp-notifications'));
        }
        
        $phone = sanitize_text_field($_POST['phone'] ?? '');
        $message = sanitize_textarea_field($_POST['message'] ?? '');
        
        if (empty($phone) || empty($message)) {
            wp_send_json_error(__('رقم واتساب ونص الرسالة مطلوبان', 'woocommerce-whatsapp-notifications'));
        }
        
        $settings = get_option('wwnt_settings');
        $from_number = $settings['from_number'] ?? '';
        
        if (empty($from_number)) {
            wp_send_json_error(__('رقم الإرسال غير مُعد', 'woocommerce-whatsapp-notifications'));
        }
        
        $result = $this->send_message($phone, $from_number, $message);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success(__('تم إرسال رسالة الأختبار بنجاح', 'woocommerce-whatsapp-notifications'));
        }
    }
}
