<?php
if (!defined('ABSPATH')) exit;

class WWNT_API_Handler {
    private static $instance = null;
    private $api_url;
    private $token;

    private function __construct() {
    // 🔒 نجبر الاضافة تشتغل على الرابط المحدد فقط
    $this->api_url = 'https://bot.watsira.com/api/qr';

    // 🔑 التوكن يبقى من الاعدادات (او تحطه ثابت لو تبي)
    $settings    = get_option('wwnt_settings');
    $this->token = isset($settings['token']) ? $settings['token'] : '';
}


    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * نقطة إرسال موحّدة (QR فقط)
     */
    public function send_whatsapp_message($to, $message_type, $message_data) {
        if (empty($this->api_url) || empty($this->token)) {
            return new WP_Error('api_config_missing', __('API URL or Token is missing', 'woocommerce-whatsapp-notifications'));
        }

        $message_data = $this->sanitize_message_data($message_data);
        return $this->send_qr_message($to, $message_type, $message_data);
    }

    /**
     * 🔧 تطبيع رقم واتساب للاستخدام مع واجهة الإرسال:
     * - يزيل محارف التحكم والمسافات والرموز (ما عدا + في البداية)
     * - يدعم مدخلات مثل: +966 50 521 2879 / 00966505212879 / 0505212879 / 966505212879
     * - لا يكرر رمز الدولة إذا كان الرقم يبدأ به أصلاً
     * - تُعيد الرقم بدون علامة + لأن الـ API يتوقعه بدون +
     */
    private function normalize_msisdn($number) {
        // إزالة محارف تحكم خفية (مثل LRM/RLM)
        $number = preg_replace('/[\x{200E}\x{200F}\x{202A}-\x{202E}\x{2066}-\x{2069}]/u', '', (string)$number);
        $number = trim($number);

        // السماح بـ + في البداية ثم أرقام فقط، إزالة باقي الرموز/المسافات
        $number = preg_replace('/(?!^\+)\D+/', '', $number);

        // قراءة رمز الدولة الافتراضي (أرقام فقط)
        $options      = get_option('wwnt_settings');
        $default_code = isset($options['default_country_code']) ? preg_replace('/\D+/', '', $options['default_country_code']) : '';
        if ($default_code === '') {
            $default_code = '966';
        }

        // 00 إلى +
        if (strpos($number, '00') === 0) {
            $number = '+' . substr($number, 2);
        }

        // إذا يبدأ بـ +
        if (strpos($number, '+') === 0) {
            // أرجع الرقم بدون +
            return ltrim($number, '+');
        }

        // الآن الرقم أرقام فقط
        // إذا بدأ بـ 0 → اعتبره وطني: استبدل 0 بـ رمز الدولة
        if (strpos($number, '0') === 0) {
            return $default_code . substr($number, 1);
        }

        // إذا بدأ بالفعل برمز الدولة الافتراضي، لا تضف مرة ثانية
        if (strpos($number, $default_code) === 0) {
            return $number; // فعليًا رقم دولي بدون +
        }

        // قرار بناءً على الطول:
        // - لو قصير (<=10) اعتبره وطني وأضف الرمز
        // - لو طويل، اعتبره يتضمن رمز دولة (اتركه كما هو)
        $len = strlen($number);
        if ($len <= 10) {
            return $default_code . $number;
        }

        return $number;
    }

    /**
     * إرسال عبر خادم QR
     */
    private function send_qr_message($to, $message_type, $message_data) {
        $from = isset($message_data['from']) ? $message_data['from'] : '';

        // ✅ تطبيع الأرقام
        $to   = $this->normalize_msisdn($to);
        $from = $from ? $this->normalize_msisdn($from) : '';

        $params = [
            'messageType' => $message_type,
            'requestType' => 'POST',
            'token'       => $this->token,
            'from'        => $from,
            'to'          => $to,
            'text'        => $message_data['text'] ?? ''
        ];

        $this->log_request("QR API Request: " . json_encode($params, JSON_UNESCAPED_UNICODE));

        $response = wp_remote_post(rtrim($this->api_url, '/') . '/rest/send_message', [
            'body'        => $params,
            'timeout'     => 45,
            'httpversion' => '1.1',
            'headers'     => ['Content-Type' => 'application/x-www-form-urlencoded'],
        ]);

        return $this->handle_response($response);
    }

    /**
     * التعامل مع استجابة API
     */
    private function handle_response($response) {
        if (is_wp_error($response)) {
            $this->log_request("Error: " . $response->get_error_message());
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        $this->log_request("API Response: " . json_encode($data, JSON_UNESCAPED_UNICODE));

        if (isset($data['success']) && $data['success']) {
            return $data;
        } else {
            $error_message = $data['message'] ?? __('Unknown error occurred or check if the API endpoints are correct', 'woocommerce-whatsapp-notifications');
            $this->log_request("API Error: " . $error_message);
            return new WP_Error('api_error', $error_message, $data);
        }
    }

    /**
     * تنظيف بيانات الرسالة (حماية بسيطة)
     */
private function sanitize_message_data($message_data) {
    if (!is_array($message_data)) {
        return $message_data;
    }

    foreach ($message_data as $key => $value) {

        // 1️⃣ تأكد أنه نص
        $value = (string) $value;

        // 2️⃣ فك أي HTML entities (يعالج &amp;)
        $value = html_entity_decode(
            $value,
            ENT_QUOTES | ENT_HTML5,
            'UTF-8'
        );

        // 3️⃣ توحيد الأسطر (يحافظ عليها)
        $value = str_replace(
            ["\r\n", "\r"],
            "\n",
            $value
        );

        // 4️⃣ إزالة أي وسوم HTML يدويًا (بدون touch للنص)
        // ⚠️ لا wp_kses ولا wp_strip_all_tags
        $value = preg_replace('/<[^>]*>/', '', $value);

        // 5️⃣ منع تكدس الأسطر فقط
        $value = preg_replace("/\n{3,}/", "\n\n", $value);

        $message_data[$key] = trim($value);
    }

    return $message_data;
}


    /**
     * لوج محلي
     */
    private function log_request($message) {
        $log_file = ABSPATH . 'wp-content/log.txt';
        @file_put_contents($log_file, '['.date('Y-m-d H:i:s').'] ' . $message . "\n", FILE_APPEND);
    }
}

/**
 * ✅ تطبيع رقم واتساب وتخزينه بصيغة E.164 (مع +) عند حفظ الطلب (Checkout + لوحة التحكم)
 * - هذا للتخزين فقط (يعرض للمدير رقم نظيف)
 * - أثناء الإرسال، الدالة داخل الكلاس تُرجع الرقم بدون +
 */
add_action('woocommerce_checkout_update_order_meta', 'wwnt_normalize_order_phone_for_storage');
add_action('save_post_shop_order', 'wwnt_normalize_order_phone_for_storage');

function wwnt_normalize_order_phone_for_storage($order_id) {
    if (empty($order_id)) return;

    $order = wc_get_order($order_id);
    if (!$order) return;

    $phone = $order->get_billing_phone();
    if (empty($phone)) return;

    $normalized = wwnt_normalize_phone_number_for_storage($phone);

    if (!empty($normalized) && $normalized !== $phone) {
        $order->set_billing_phone($normalized);
        $order->save();
    }
}

/**
 * 🔧 تطبيع رقم واتساب للتخزين (يعيده مع علامة +)
 */
function wwnt_normalize_phone_number_for_storage($number) {
    // 🧹 إزالة محارف التحكم الخفية
    $number = preg_replace('/[\x{200E}\x{200F}\x{202A}-\x{202E}\x{2066}-\x{2069}]/u', '', (string)$number);
    $number = trim($number);

    // 🧹 إزالة كل الرموز ما عدا + في البداية والأرقام
    $number = preg_replace('/(?!^\+)\D+/', '', $number);

    // 00 -> +
    if (strpos($number, '00') === 0) {
        $number = '+' . substr($number, 2);
    }

    // ⚙️ إعدادات
    $options      = get_option('wwnt_settings');
    $default_code = isset($options['default_country_code']) ? preg_replace('/\D+/', '', $options['default_country_code']) : '';
    if ($default_code === '') {
        $default_code = '966'; // السعودية افتراضي
    }

    // ✅ لو يبدأ بـ +
    if (strpos($number, '+') === 0) {
        return $number; // جاهز للتخزين
    }

    // ✅ لو يبدأ بـ 0 → محلي
    if (strpos($number, '0') === 0) {
        return '+' . $default_code . substr($number, 1);
    }

    // ✅ لو يبدأ برمز الدولة الافتراضي مباشرة
    if (strpos($number, $default_code) === 0) {
        return '+' . $number;
    }

    // ✅ حسب الطول
    $len = strlen($number);
    if ($len >= 7 && $len <= 10) {
        // قصير → محلي
        return '+' . $default_code . $number;
    }

    // ✅ دولي بدون + (رمز دولة آخر)
    return '+' . $number;
}
